#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <time.h>

// Urceni pozice na primce metodou puleni intervalu
// KSI 2013/2014: 3-1 "Zapomenuty poklad"
// Jan Horacek
// 4.1.2014
// v1.0

// tento algoritmus se vzdy zepta papouska na 2 body ve 2/3 a 1/3 usecky
//  a z toho urci, v jake polovine se nachazi poklad
//  pro tuto polovinu se pak rekurzivne zavola
//  viz reseni ulohy v pdf (poklad.pdf)

//#define DEBUG                             // pro zapnuti DEBUG vypisu odkomentuj

///////////////////////////////////////////////////////////

unsigned int last_pos;                      // posledni pozice, na kterou jsme se ptali papouska
unsigned int poklad_pos;                    // pozice pokladu

///////////////////////////////////////////////////////////

// emuluje papouska
// vraci true, pokud jsme bliz pokladu (poklad_pos), jinak false
bool papousek(unsigned int pos)
{
    if (abs(pos-poklad_pos) < abs(last_pos-poklad_pos))
    {
        last_pos = pos;
        return true;
    } else {
        last_pos = pos;
        return false;
    }
}

///////////////////////////////////////////////////////////

// pokud zbyvaji posledni tri, zavolame tuto fci
// tj. left, left+1, left+2
// vola se proto, ze 3 prvky se spatne puli
int tri(int left)
{
    papousek(left);
    if (!papousek(left+1)) return left;
    if (papousek(left+2)) return left+2; else return left+1;
}

///////////////////////////////////////////////////////////

// tyto definice slouzi s tzv. "parrot-hack"u, ktery slouzi k minimalnizace pristupu k papouskovi
//  viz poklad.pdf
// myslenkou je pamatovani si posledni ctene pozice vuci aktualni rekurzivni "iteraci" funkce
// pokud tedy funkce napr. dostane na vstupu hodnotu LAST_L, vi, ze posledni pozadavek na papouska byl v 1/3 jejiho inetrvalu
//  tudiz se zepta jen na 2/3 a z toho vi, v ktere polovine je poklad
//  : neni nutne se ptat 2x
#define LAST_N  0           // posledni cteni neni pouzitelne
#define LAST_L  1           // posledni cteni bylo vlevo (1/3)
#define LAST_R  2           // posledni cteni bylo vpravo (2/3)

// hlavni rekurzivni prohledavaci fce
// na vstupu dostane hranice - podobne, jako napr. quicksort a flag last_read pro parrot-hack
// v teto fci je impelmentovani nromalni zaokrouhlavani - ne oseknuti desetinne casti
int search(int left_limit, int right_limit, int last_read)
{
    // ptame se na 2 body ve tretinach mezi left_limit a right_limit
    // dovizme se, jestli poklad je v leve polovine intervalu, nebo v prave

    // cteme vzdy nejdriv pravou a pak levou !

    bool vlevo;                     // je poklad vlevo? pokud ne, je vpravo
    int new_read;                   // s timthletim flagem parrot-hack volam rekurzivni funkci, promenna je tady proto, abych si jeho hodnotu predpripravil

    #ifdef DEBUG
    printf(">>> Ctu interval: %d - %d\n", left_limit, right_limit);
    #endif

    // pokud zbyvaji posledni tri
    if ((right_limit - left_limit) == 2)
        return tri(left_limit);

    // pokud je last_read = LAST_L -> posledni ctenou hodnotou byla aktualni leva hodnota
    // pokud je last_read = LAST_R -> posledni ctenou hodnotou byla aktualni prava hodnota
    // pokud je last_read = LAST_N -> posledni ctenou hodnotou bylo neco jineho, musime cist oboji

    if (last_read == LAST_N)
    {
        // pokud posledni cteni nebylo, je potreba provest obe cteni

        #ifdef DEBUG
        printf(">>>>>> Posledni cteni je irelevantni\n>>>>>> Ctu %d, %d\n", (int)(((right_limit-left_limit)*(2.0/3))+left_limit+0.5), (int)(((right_limit-left_limit)/3.0)+left_limit+0.5));
        #endif

        // nejdriv vpravo
        papousek(((right_limit-left_limit)*(2.0/3.0))+left_limit+0.5);

        // pak vlevo
        if (papousek(((right_limit-left_limit)/3.0)+left_limit+0.5))
        {
            vlevo = true;
            new_read = LAST_R;         // pro dalsi rekurzivni beh bude posledni prectena hodnota vpravo
        } else {
            vlevo = false;
            new_read = LAST_N;         // pro dalsi rekurzivni beh bude posledni prectena hodnota irelevantni
        }
    }

    if (last_read == LAST_L)
    {
        #ifdef DEBUG
        printf(">>>>>> Posledni cteni bylo vlevo\n>>>>>> Ctu 2/3: %d\n", (int)(((right_limit-left_limit)*(2.0/3))+left_limit+0.5));
        #endif

        // posledni cteni bylo vlevo -> ctu vpravo
        // pokud je vpravo bliz, poklad je vpravo -> vlevo = false
        if (papousek((right_limit-left_limit)*(2.0/3)+left_limit+0.5))
        {
            vlevo = false;
            new_read = LAST_L;
        } else {
            vlevo = true;
            new_read = LAST_N;
        }
    }

    if (last_read == LAST_R)
    {
        #ifdef DEBUG
        printf(">>>>>> Posledni cteni bylo vpravo\n>>>>>> Ctu 1/3: %d\n", (int)(((right_limit-left_limit)/3.0)+left_limit+0.5));
        #endif

        // posledni cteni bylo vpravo -> ctu vlevo
        // pokud je vlevo bliz, je poklad vlevo
        if (papousek(((right_limit-left_limit)/3.0)+left_limit+0.5))
        {
            vlevo = true;
            new_read = LAST_R;
        } else {
            vlevo = false;
            new_read = LAST_N;
        }
    }

    #ifdef DEBUG
    if (vlevo)
        printf(">>>>>> Data jsou vlevo\n"); else printf(">>>>>> Data jsou vpravo\n");
    #endif

    if (vlevo)
    {
        // pokud mame 2 posledni prvky
        if (right_limit-left_limit == 1) return left_limit;

        // poklad je vlevo -> cteme dal doleva
        return search(left_limit, ((right_limit-left_limit)/2)+left_limit, new_read);
    } else {
        // pokud mame 2 posledni prvky
        if (right_limit-left_limit == 1) return right_limit;

        // poklad je vpravo -> cteme dal doprava
        return search(((right_limit-left_limit)/2)+left_limit, right_limit, new_read);
    }
}

///////////////////////////////////////////////////////////

#define MAX_LEN 0xFFFF
int main()
{
    int LEN, pos;

    srand(time(NULL));
    LEN = rand()%MAX_LEN + 1;
    poklad_pos = rand()%LEN;

    printf("size: %d, poklad: %d\n", LEN, poklad_pos);

    papousek(LEN*(2.0/3)+0.5);            // prvni volani provedeme rucne
    pos = search(0, LEN, LAST_R);

    printf("----- poklad nalezen na pozici %d\n", pos);

    #ifdef DEBUG
    if (poklad_pos != pos)
    {
        printf("MISMATCH: %d\n", pos);
        return 1;
    }
    #endif

    return 0;
}

///////////////////////////////////////////////////////////
